/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* SMP management interface */
/* Begun by Stig Telfer, Alpha Processor Inc, 30 June 1999 */

#undef TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "smp.h"
#include "specifics.h"
#include "northbridge.h"
#include "platform.h"
#include "cpu.h"


#undef	DEBUG_ONE_CPU		/* Simulate only one CPU for debug */

/*----------------------------------------------------------------------*/
/* NOTE: Due to the way Tsunami SMP systems start up, it may be random which 
 * CPU is the primary CPU.  Use the smp_primary() function and don't assume 
 * the primary is CPU 0! */

Action_t smp_box[MAX_CPUS];		/* bss: starts off zeroed */

/* smp_primary_cpu: ID of boot CPU, all others secondary */
/* The special value of '-1' means it has not yet been determined which CPU
 * is primary.  If smp_primary() is called while smp_primary_cpu is not 
 * determined, it is safe to assume that the caller is the primary CPU.
 */
volatile int smp_primary_cpu = NOT_DETERMINED;

smp_mask smp_all_cpus = 0;
volatile unsigned char smp_online_count = 0;	


/*----------------------------------------------------------------------*/
/* SMP Mutex primitives */

void smp_acquire( smp_mutex *M )
{
    smp_mask mval = 1<<smp_phys_id();

    /* first check to see if I already hold this mutex */
    if ( *M == mval )		return;

    /* use atomic memory operations to ensure exclusion of other cpus */
    atomic_tas( M, mval );
}

void smp_release( smp_mutex *M )
{
    *M = M_FREE;		/* mark it as being available for other CPUs */
    asm("mb");
    usleep( 1 );		/* allows a probe, injects a little fairness */
}



/*----------------------------------------------------------------------*/
/* SMP processor enumeration functions */

int smp_ncpus( void )
{
    return smp_online_count;
}

int smp_phys_id( void )
{
#ifdef DEBUG_ONE_CPU	
    return 0;
#else
    /* hopefully there will never be a system where the northbridge doesn't 
     * know the SMP details... */
    return nb_whami();			
#endif	/* DEBUG_ONE_CPU */
}

static uint8 smp_id_table[ MAX_CPUS ];

int smp_myid( void )
{
    int phys_id = smp_phys_id();
    smp_mask m = 1 << phys_id;

    /* Does this processor not have a logical ID yet? */
    if ( (smp_all_cpus & m) == 0 )
    {
	mobo_logf(
	    LOG_WARN "SMP: CPU %d invoking SMP interface before registering\n"
	    LOG_WARN "SMP: %d CPUs registered so far (mask %X)\n",
		phys_id, smp_online_count, smp_all_cpus );
	smp_register();
    }

    return smp_id_table[ phys_id ];
}


/* NB this is important as an early test, and should not require a call to
 * smp_register first
 */
BOOLEAN smp_primary( void )
{
    if ( smp_primary_cpu == NOT_DETERMINED )
	return TRUE;

    return ( smp_phys_id() == smp_primary_cpu );
}


BOOLEAN smp_register( void )
{
    int phys_id;
    smp_mask m;
    static smp_mutex smp_reg = M_FREE;

    phys_id = smp_phys_id( );
    m = 1 << phys_id;

    /* Have we already registered? */
    if ( smp_all_cpus & m )
	return FALSE;

    /* ---- Begin critical region ---- */ 
    smp_acquire( &smp_reg );

    if ( smp_online_count == 0 )
	smp_primary_cpu = phys_id;

    smp_all_cpus |= m;

    smp_id_table[ smp_online_count ] = phys_id;
    smp_online_count++;

    smp_release( &smp_reg );
    /* ---- End critical region ---- */ 

    return TRUE;
}



/*----------------------------------------------------------------------*/
/* SMP synchronisation primitives */

#define NO_CPUS_YET -1

/* Implementation note: the timer only ticks if interrupts are enabled.
 * which suggests we're running the memory tests or some other stress test
 */
#define CPU_POLL_TIMEOUT    4096	/* in 1024-Hz ticks */

void smp_sync( void )
{
    static smp_mutex sync_entry = M_FREE;
    static smp_mutex sync_exit = M_FREE;
    static smp_mask entry_mask = 0, exit_mask = 0;
    int phys_id = smp_phys_id();
    static int first_cpu_in = NO_CPUS_YET;
    unsigned ejiffies=0;
    int warned = FALSE;

    smp_acquire( &sync_entry );
    entry_mask |= 1 << phys_id;
    if ( first_cpu_in == NO_CPUS_YET )
    {
	first_cpu_in = phys_id;		/* I'm hosting this sync */
	ejiffies = jiffies + CPU_POLL_TIMEOUT;
    }

    if ( entry_mask == smp_all_cpus ) { 

	/* at this point, all CPUs have entered sync */

	/* last CPU takes controlling path */
	smp_acquire( &sync_exit );
	exit_mask |= 1 << phys_id;
	smp_release( &sync_exit );

	while( exit_mask != smp_all_cpus )	/* poll */	;

	/* at this point, all cpus have seen that all other cpus are synced */

	exit_mask = entry_mask = 0;		/* clean the slate */
	first_cpu_in = NO_CPUS_YET;

	/* another sync may now safely take place */

	smp_release( &sync_entry );
	
	/* and we're done! */
	
    } else {
	smp_release( &sync_entry );

	while( entry_mask != smp_all_cpus)
	{
	    if ( phys_id == first_cpu_in )
	    {
		/* At this point we check to see if we're waiting in vain */
		if ( jiffies >= ejiffies )
		{
		    if ( warned )	continue;
		    warned = TRUE;

		    mobo_logf(LOG_CRIT "SMP: Awaiting CPUs (bitmask) 0x%X\n",
			entry_mask ^ smp_all_cpus );
		    mobo_alertf("SMP timeout",
			"Waiting on CPUs (bitmask) 0x%X...",
			entry_mask ^ smp_all_cpus );
		}
	    }
	}

	smp_acquire( &sync_exit );
	exit_mask |= 1 << phys_id;		/* show we've seen all synced */
	smp_release( &sync_exit );

	/* and we're done! */
    }
}



/*----------------------------------------------------------------------*/
/* SMP secondary CPU holding function */

void smp_corral( void )
{
    unsigned phys_id = smp_phys_id();
    Action_t myjob;
    DBM_STATUS sval;

    /* transfer the setup for uart over from primary */
    impure[ phys_id ]->AUTOBAUD = primary_impure->AUTOBAUD;

    /* Register processor information */
    cpu_info_submit( phys_id );


#if 0
    smp_acquire( &disp_lock );

    mobo_alertf( "Processor On-line", "Hello from CPU %d: I am %s\n",
		phys_id, smp_primary() ? "primary" : "secondary" );
    plat_infodump();

    smp_release( &disp_lock );
#endif

    mobo_logf( LOG_INFO "CPU %d on-line and ready for action\n", phys_id );

    /* Secondary CPU event loop.
     * A job is signalled for secondary CPUs by letterboxing */

    swpipl( 7 );			/* initialise secondary interrupts */

    while ( TRUE )
    {
	/* Is there a job for us? */
	while ( (myjob = smp_box[phys_id]) == NULL )
	{
	    msleep( 10 );		/* don't loop too furiously */
	}
	smp_box[phys_id] = NULL;	/* I have collected the message */

	TRACE( "SMP: CPU %d joining parallel job at 0x%X\n", phys_id, myjob );

	sval = myjob( 0, NULL );	/* perform the action */
	mobo_logf( LOG_INFO "SMP: CPU %d returned %s from parallel job\n",
		phys_id, (sval == STATUS_SUCCESS) ? "success" : "failure" );
    }
}



/*----------------------------------------------------------------------*/
/* SMP parallelised function call */

DBM_STATUS smp_exec( Action_t A )
{
    int i;

    /* send the address to all CPUs present via their letterboxes */
    /* actually the policy here is to write to all CPUs, present or not,
     * which makes things simpler and does no harm */

    for ( i=0; i < MAX_CPUS; i++ )
	smp_box[i] = A;

    /* now make the jump on the primary */
    return A( 0, NULL );
}


/*----------------------------------------------------------------------*/
/* SMP Inter-processor signalling (IPI) */

typedef struct _ipistruct {
	unsigned char source;
	IPI_t type;
	uint64 arg;
} IPI_queue_t;

static IPI_queue_t cpu_queue[MAX_CPUS];
static smp_mutex cpu_queue_mutex[MAX_CPUS];

volatile unsigned long smp_ipi_pingpongs = 0;
static smp_mutex pingpong_mutex = M_FREE;

static DBM_STATUS queue_on_processor( unsigned cpu, IPI_t type, uint64 arg )
{
    int i;
    static unsigned char unused = TRUE;

    if ( unused )		/* perform data structure initialisation */
    {
	for ( i = 0; i < MAX_CPUS; i++ )
	{
	    cpu_queue[i].source = 0,
	    cpu_queue[i].type = IPI_NULL,
	    cpu_queue[i].arg = 0,
	    cpu_queue_mutex[i] = M_FREE;
	}
	unused = FALSE;
    }

    /* CPU is already handling an IPI? */
    if ( cpu_queue[cpu].type != IPI_NULL )		
	return STATUS_FAILURE;

    /* ------ acquire mutual exclusion of this queue ------ */
    smp_acquire( &cpu_queue_mutex[cpu] );

    cpu_queue[cpu].source = smp_phys_id();
    cpu_queue[cpu].type = type,
    cpu_queue[cpu].arg = arg;

    smp_release( &cpu_queue_mutex[cpu] );
    /* ------ release mutual exclusion of this queue ------ */

    return STATUS_SUCCESS;
}

static void dequeue_from_processor( unsigned cpu, IPI_queue_t *T )
{
    /* Fill in the details from queue bottom */
    /* ------ acquire mutual exclusion of this queue ------ */
    smp_acquire( &cpu_queue_mutex[cpu] );

    memcpy( T, &cpu_queue[cpu], sizeof( IPI_queue_t ) );

    /* Since the current queue implementation has only 1 member,
     * shuffling things up merely involves wiping out the last posting */

    cpu_queue[cpu].source = 0,
    cpu_queue[cpu].type = IPI_NULL,
    cpu_queue[cpu].arg = 0;

    smp_release( &cpu_queue_mutex[cpu] );
    /* ------ release mutual exclusion of this queue ------ */

}


DBM_STATUS smp_ipi( unsigned cpu, IPI_t type, uint64 arg )
{
    DBM_STATUS sval;

    /* Queue the request on the specified processor */
    sval = queue_on_processor( cpu, type, arg );
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "IPI: queueing failed for IPI request on CPU %d!\n",
		    cpu );
	return STATUS_FAILURE;
    }

    /* now give the specified processor a ding-dong */
    wripir( cpu );
    return STATUS_SUCCESS;
}



/* Called through the interrupt handler dispatch routine */

void smp_ipi_handler( void )
{
    IPI_queue_t ipidata;
    int phys_id = smp_phys_id();

    /* de-queue the request information left by the sender */
    dequeue_from_processor( phys_id, &ipidata );

    switch( ipidata.type )
    {
	case IPI_NULL:		/* something really weird is happening! */
	    mobo_logf( LOG_CRIT
		       "IPI: Bizarre!  Received IPI with no data queued\n" );
	    break;

	case IPI_PING:		/* recipient should reply to a ping request */
#ifdef CONFIG_TINOSA
	    /* TINOSA DEBUG */
	    outportb( 0x92, 0x00 );	/* this globally deasserts the IPI */
#endif
	    TRACE( "IPI: ping...\n" );
	    smp_ipi( ipidata.source, IPI_PONG, 0 );
	    break;

	case IPI_PONG:		/* response from ping - did I ping them? */
#ifdef CONFIG_TINOSA
	    /* TINOSA DEBUG */
	    outportb( 0x92, 0x00 );	/* this globally deasserts the IPI */
#endif
	    TRACE( "IPI: pong...\n" );
	    smp_acquire( &pingpong_mutex );
	    smp_ipi_pingpongs++;
	    smp_release( &pingpong_mutex );
	    break;

	case IPI_EXEC:
	default:
	    mobo_alertf( "Unknown/unsupported IPI call!",
			 "Sorry, please come back when I've written this code");
	    break;
    }
}
